/**********************************************************************************************************************
 Thanks for running these splitter tests for me.

 This script is a mostly hands off script.  It does everything needed for the tests.  All you need to do is...
 1.  Please make sure that SSMS is in the "Grid" output mode when running this script.
 2.  Run this script.  It builds all of the objects it needs and produces the result set need at the end.
     Note that this script runs in TempDB so as not to take any chances with your data, sprocs, etc.
 3.  When the run is Please copy the second result set (including the column names) into a spreadsheet and
     sent the spreadsheet to me at jbmoden@ameritech.net.
 4.  This script also deletes all of the objects it created except for the final result set which can
     be deleted after you've sent me the spreadsheet.  I left it there so if something went wrong,
     you wouldn't have to rerun the whole test again.

It would also be helpful if you provided a brief description of your hardware and the version of SQL Server that
your running on.  This script does NOT automatically capture any information about your machine.

 Thanks again for your help.
 --Jeff Moden - 10 Apr 2011
**********************************************************************************************************************/

--===== Do this all in a nice, safe place that everyone has.
    USE TempDB;
GO
--=====================================================================================================================
--      Conditionally drop and recreate the "SplitInchworm" splitter function
--=====================================================================================================================
     IF OBJECT_ID('dbo.SplitInchworm') IS NOT NULL
        DROP FUNCTION dbo.SplitInchworm;
GO
 CREATE FUNCTION dbo.SplitInchWorm
        (
        @pString    VARCHAR(8000),
        @pDelimiter CHAR(1)
        )
RETURNS @Return TABLE (ItemNumber INT, Item VARCHAR(8000))
     AS
  BEGIN
        DECLARE @StartPointer INT,
                @EndPointer   INT,
                @Counter      INT;

         SELECT @StartPointer = 1,
                @EndPointer   = CHARINDEX(@pDelimiter, @pString), 
                @Counter      = 1;

          WHILE @EndPointer > 0
          BEGIN 
                 INSERT INTO @Return
                        (ItemNumber, Item)  
                 SELECT ItemNumber = @Counter,
                        Item       = SUBSTRING(@pString, @StartPointer, @EndPointer - @StartPointer);

                 SELECT @StartPointer = @EndPointer + 1,
                        @EndPointer   = CHARINDEX(@pDelimiter, @pString, @StartPointer),
                        @Counter      = @Counter + 1;
            END;

 INSERT INTO @Return
        (ItemNumber, Item)  
 SELECT ItemNumber = @Counter,
        Item       = SUBSTRING(@pString, @StartPointer, 8000);

 RETURN;
    END
GO
--=====================================================================================================================
--      Conditionally drop and recreate the "SplitNibbler" splitter function
--=====================================================================================================================
     IF OBJECT_ID('dbo.SplitNibbler') IS NOT NULL
        DROP FUNCTION dbo.SplitNibbler;
GO
 CREATE FUNCTION dbo.SplitNibbler
        (
        @pString    VARCHAR(8000),
        @pDelimiter CHAR(1)
        )
RETURNS @Return TABLE (ItemNumber SMALLINT, Item VARCHAR(8000))
   WITH SCHEMABINDING AS
  BEGIN
        DECLARE @EndPointer INT,
                @Counter    INT;

         SELECT @EndPointer = CHARINDEX(@pDelimiter, @pString), 
                @Counter    = 1;

          WHILE @EndPointer > 0
          BEGIN 
                 INSERT INTO @Return
                        (ItemNumber, Item)
                 SELECT ItemNumber  = @Counter,
                        Item        = SUBSTRING(@pString, 1, @EndPointer - 1);

                 SELECT @pString    = SUBSTRING(@pString, @EndPointer + 1, DATALENGTH(@pString)),
                        @EndPointer = CHARINDEX(@pDelimiter, @pString),
                        @Counter    = @Counter + 1;
            END;

 INSERT INTO @Return
        (ItemNumber, Item)
 SELECT ItemNumber = @Counter,
        Item       = @pString;

 RETURN;
    END;
GO
--=====================================================================================================================
--      Conditionally drop and recreate the "SplitRCte" splitter function
--=====================================================================================================================
     IF OBJECT_ID('dbo.SplitRCte') IS NOT NULL
        DROP FUNCTION dbo.SplitRCte;
GO
 CREATE FUNCTION dbo.SplitRCte
        (
        @pString VARCHAR(8000),
        @pDelimiter CHAR(1)
        )
RETURNS TABLE
   WITH SCHEMABINDING AS
 RETURN
   WITH cteSplit AS
(
 SELECT StartPosition = 0,
        EndPosition   = CHARINDEX(@pDelimiter,@pString),
        ItemNumber    = 1
  UNION ALL
 SELECT EndPosition + 1,
        CHARINDEX(@pDelimiter, @pString, EndPosition + 1), 
        ItemNumber + 1
   FROM cteSplit
  WHERE EndPosition > 0
)
 SELECT ItemNumber,
        Item = SUBSTRING(@pString,StartPosition,ISNULL(NULLIF(EndPosition,0),8000)-StartPosition)
   FROM cteSplit;
GO
--=====================================================================================================================
--      Conditionally drop and recreate the "SplitXML" splitter function
--=====================================================================================================================
     IF OBJECT_ID('dbo.SplitXML') IS NOT NULL
        DROP FUNCTION dbo.SplitXML;
GO
 CREATE FUNCTION dbo.SplitXML
        (
        @pString    VARCHAR(MAX), 
        @pDelimiter CHAR(1)
        )
RETURNS @Result TABLE (ItemNumber INT, Item VARCHAR(8000)) 
   WITH SCHEMABINDING AS
  BEGIN
        DECLARE @XML XML;

         SELECT @XML = '<r>' + REPLACE(@pString, ',', '</r><r>') + '</r>';

         INSERT INTO @Result 
                    (ItemNumber, Item)
         SELECT ItemNumber = ROW_NUMBER() OVER (ORDER BY (SELECT NULL)),
                Item       = Item.value('text()[1]', 'varchar(8000)')
           FROM @XML.nodes('//r') R(Item);
 RETURN;
   END;
GO
--=====================================================================================================================
--      Conditionally drop and recreate the "OLDDelimitedSplit8K" splitter function
--      This is the "original" Tally Table (CTE really) which has the long-string performance problem.
--=====================================================================================================================
     IF OBJECT_ID('dbo.OLDDelimitedSplit8K') IS NOT NULL
        DROP FUNCTION dbo.OLDDelimitedSplit8K;
GO
 CREATE FUNCTION dbo.OLDDelimitedSplit8K
--===== Define I/O parameters
        (
        @pString    VARCHAR(7999),
        @pDelimiter CHAR(1)
        )
RETURNS TABLE
   WITH SCHEMABINDING AS
 RETURN
--===== "Inline" CTE Driven "Tally Table produces values up to
     -- 10,000... enough to cover VARCHAR(8000)
WITH
      E1(N) AS ( --=== Create Ten 1's
                 SELECT 1 UNION ALL SELECT 1 UNION ALL
                 SELECT 1 UNION ALL SELECT 1 UNION ALL
                 SELECT 1 UNION ALL SELECT 1 UNION ALL
                 SELECT 1 UNION ALL SELECT 1 UNION ALL
                 SELECT 1 UNION ALL SELECT 1 --10
               ),
      E2(N) AS (SELECT 1 FROM E1 a, E1 b),   --100
      E4(N) AS (SELECT 1 FROM E2 a, E2 b),   --10,000
cteTally(N) AS (SELECT ROW_NUMBER() OVER (ORDER BY (SELECT N)) FROM E4)  
--===== Do the split
 SELECT ItemNumber = ROW_NUMBER() OVER (ORDER BY N),
        Item       = SUBSTRING(@pString, N, CHARINDEX(@pDelimiter, @pString + @pDelimiter, N) - N)
   FROM cteTally
  WHERE N <= DATALENGTH(@pString)+1 --DATATLENGTH allows for trailing space delimiters
    AND SUBSTRING(@pDelimiter + @pString, N, 1) = @pDelimiter
;
GO
--=====================================================================================================================
--      Conditionally drop and recreate the NEW optimized "DelimitedSplit8K" splitter function.
--=====================================================================================================================
     IF OBJECT_ID('dbo.DelimitedSplit8K') IS NOT NULL
        DROP FUNCTION dbo.DelimitedSplit8K;
GO
CREATE FUNCTION [dbo].[DelimitedSplit8K]
/**********************************************************************************************************************
 Purpose:
 Split a given string at a given delimiter and return a list of the split elements (items).

 Notes:
 1.  Leading a trailing delimiters are treated as if an empty string element were present.
 2.  Consecutive delimiters are treated as if an empty string element were present between them.
 3.  Except when spaces are used as a delimiter, all spaces present in each element are preserved.

 Returns:
 iTVF containing the following:
 ItemNumber = Element position of Item as a BIGINT (not converted to INT to eliminate a CAST)
 Item       = Element value as a VARCHAR(8000)

 CROSS APPLY Usage Examples and Tests:
--=====================================================================================================================
-- TEST 1:
-- This tests for various possible conditions in a string using a comma as the delimiter.  The expected results are
-- laid out in the comments
--=====================================================================================================================
--===== Conditionally drop the test tables to make reruns easier for testing.
     -- (this is NOT a part of the solution)
     IF OBJECT_ID('tempdb..#JBMTest') IS NOT NULL DROP TABLE #JBMTest
;
--===== Create and populate a test table on the fly (this is NOT a part of the solution).
     -- In the following comments, "b" is a blank and "E" is an element in the left to right order.
     -- Double Quotes are used to encapsulate the output of "Item" so that you can see that all blanks
     -- are preserved no matter where they may appear.
 SELECT *
   INTO #JBMTest
   FROM (                                               --# & type of Return Row(s)
         SELECT  0, NULL                      UNION ALL --1 NULL
         SELECT  1, SPACE(0)                  UNION ALL --1 b (Empty String)
         SELECT  2, SPACE(1)                  UNION ALL --1 b (1 space)
         SELECT  3, SPACE(5)                  UNION ALL --1 b (5 spaces)
         SELECT  4, ','                       UNION ALL --2 b b (both are empty strings)
         SELECT  5, '55555'                   UNION ALL --1 E
         SELECT  6, ',55555'                  UNION ALL --2 b E
         SELECT  7, ',55555,'                 UNION ALL --3 b E b
         SELECT  8, '55555,'                  UNION ALL --2 b B
         SELECT  9, '55555,1'                 UNION ALL --2 E E
         SELECT 10, '1,55555'                 UNION ALL --2 E E
         SELECT 11, '55555,4444,333,22,1'     UNION ALL --5 E E E E E 
         SELECT 12, '55555,4444,,333,22,1'    UNION ALL --6 E E b E E E
         SELECT 13, ',55555,4444,,333,22,1,'  UNION ALL --8 b E E b E E E b
         SELECT 14, ',55555,4444,,,333,22,1,' UNION ALL --9 b E E b b E E E b
         SELECT 15, ' 4444,55555 '            UNION ALL --2 E (w/Leading Space) E (w/Trailing Space)
         SELECT 16, 'This,is,a,test.'                   --E E E E
        ) d (SomeID, SomeValue)
;
--===== Split the CSV column for the whole table using CROSS APPLY (this is the solution)
 SELECT test.SomeID, test.SomeValue, split.ItemNumber, Item = QUOTENAME(split.Item,'"')
   FROM #JBMTest test
  CROSS APPLY dbo.DelimitedSplit8k(test.SomeValue,',') split
;
--=====================================================================================================================
-- TEST 2:
-- This tests for various "alpha" splits and COLLATION using all ASCII characters from 0 to 255 as a delimiter against
-- a given string.  Note that not all of the delimiters will be visible and some will show up as tiny squares because
-- they are "control" characters.  More specifically, this test will show you what happens to various non-accented 
-- letters for your given collation depending on the delimiter you chose.
--=====================================================================================================================
WITH 
cteBuildAllCharacters (String,Delimiter) AS 
(
 SELECT TOP 256 
        'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789',
        CHAR(ROW_NUMBER() OVER (ORDER BY (SELECT NULL))-1)
   FROM master.sys.all_columns
)
 SELECT ASCII_Value = ASCII(c.Delimiter), c.Delimiter, split.ItemNumber, Item = QUOTENAME(split.Item,'"')
   FROM cteBuildAllCharacters c
  CROSS APPLY dbo.DelimitedSplit8k(c.String,c.Delimiter) split
  ORDER BY ASCII_Value, split.ItemNumber
;
-----------------------------------------------------------------------------------------------------------------------
 Other Notes:
 1. Optimized for VARCHAR(8000) or less.  No testing or error reporting for truncation at 8000 characters is done.
 2. Optimized for single character delimiter.  Multi-character delimiters should be resolvedexternally from this 
    function.
 3. Optimized for use with CROSS APPLY.
 4. Does not "trim" elements just in case leading or trailing blanks are intended.
 5. If you don't know how a Tally table can be used to replace loops, please see the following...
    http://www.sqlservercentral.com/articles/T-SQL/62867/
 6. Changing this function to use VARCHAR(MAX) will cause it to run twice as slow.  It's just the nature of 
    VARCHAR(MAX) whether it fits in-row or not.
 7. Multi-machine testing for the method of using UNPIVOT instead of 10 SELECT/UNION ALLs shows that the UNPIVOT method
    is quite machine dependent and can slow things down quite a bit.
-----------------------------------------------------------------------------------------------------------------------
 Credits:
 This code is the product of many people's efforts including but not limited to the following:
 cteTally concept originally by Iztek Ben Gan and "decimalized" by Lynn Pettis (and others) for a bit of extra speed
 and finally redacted by Jeff Moden for a different slant on readability and compactness. Hat's off to Paul White for
 his simple explanations of CROSS APPLY and for his detailed testing efforts. Last but not least, thanks to
 Ron "BitBucket" McCullough and Wayne Sheffield for their extreme performance testing across multiple machines and
 versions of SQL Server. 

 I also thank whoever wrote the first article I ever saw on "numbers tables" which is located at the following URL
 and to Adam Machanic for leading me to it many years ago.
 http://sqlserver2000.databases.aspfaq.com/why-should-i-consider-using-an-auxiliary-numbers-table.html
-----------------------------------------------------------------------------------------------------------------------
 Revision History:
 Rev 00 - 20 Jan 2010 - Concept for inline cteTally: Lynn Pettis and others.
                        Redaction/Implementation: Jeff Moden 
        - Base 10 redaction and reduction for CTE.  (Total rewrite)

 Rev 01 - 13 Mar 2010 - Jeff Moden
        - Removed one additional concatenation and one subtraction from the SUBSTRING in the SELECT List for that tiny
          bit of extra speed.

 Rev 02 - 14 Apr 2010 - Jeff Moden
        - No code changes.  Added CROSS APPLY usage example to the header, some additional credits, and extra 
          documentation.

 Rev 03 - 18 Apr 2010 - Jeff Moden
        - No code changes.  Added notes 7, 8, and 9 about certain "optimizations" that don't actually work for this
          type of function.

 Rev 04 - 29 Jun 2010 - Jeff Moden
        - Added WITH SCHEMABINDING thanks to a note by Paul White.  This prevents an unnecessary "Table Spool" when the
          function is used in an UPDATE statement even though the function makes no external references.

 Rev 05 - 02 Apr 2011 - Jeff Moden
        - Rewritten for extreme performance improvement especially for larger strings approaching the 8K boundary and
          for strings that have wider elements.  The redaction of this code involved removing ALL concatenation of 
          delimiters, optimization of the maximum "N" value by using TOP instead of including it in the WHERE clause,
          and the reduction of all previous calculations (thanks to the switch to a "zero based" cteTally) to just one 
          instance of one add and one instance of a subtract. The length calculation for the final element (not 
          followed by a delimiter) in the string to be split has been greatly simplified by using the ISNULL/NULLIF 
          combination to determine when the CHARINDEX returned a 0 which indicates there are no more delimiters to be
          had or to start with. Depending on the width of the elements, this code is between 4 and 8 times faster on a
          single CPU box than the original code especially near the 8K boundary.
        - Modified comments to include more sanity checks on the usage example, etc.
        - Removed "other" notes 8 and 9 as they were no longer applicable.

 Rev 06 - 12 Apr 2011 - Jeff Moden
        - Based on a suggestion by Ron "Bitbucket" McCullough, additional test rows were added to the sample code and
          the code was changed to encapsulate the output in pipes so that spaces and empty strings could be perceived 
          in the output.  The first "Notes" section was added.  Finally, an extra test was added to the comments above.
**********************************************************************************************************************/
--===== Define I/O parameters
        (@pString VARCHAR(8000), @pDelimiter CHAR(1))
RETURNS TABLE WITH SCHEMABINDING AS
 RETURN
--===== "Inline" CTE Driven "Tally Table produces values from 0 up to 10,000...
     -- enough to cover VARCHAR(8000)
  WITH E1(N) AS (
                 SELECT 1 UNION ALL SELECT 1 UNION ALL SELECT 1 UNION ALL 
                 SELECT 1 UNION ALL SELECT 1 UNION ALL SELECT 1 UNION ALL 
                 SELECT 1 UNION ALL SELECT 1 UNION ALL SELECT 1 UNION ALL SELECT 1
                ),                          --10E+1 or 10 rows
       E2(N) AS (SELECT 1 FROM E1 a, E1 b), --10E+2 or 100 rows
       E4(N) AS (SELECT 1 FROM E2 a, E2 b), --10E+4 or 10,000 rows max
 cteTally(N) AS (--==== This provides the "zero base" and limits the number of rows right up front
                     -- for both a performance gain and prevention of accidental "overruns"
                 SELECT 0 UNION ALL
                 SELECT TOP (DATALENGTH(ISNULL(@pString,1))) ROW_NUMBER() OVER (ORDER BY (SELECT NULL)) FROM E4
                ),
cteStart(N1) AS (--==== This returns N+1 (starting position of each "element" just once for each delimiter)
                 SELECT t.N+1
                   FROM cteTally t
                  WHERE (SUBSTRING(@pString,t.N,1) = @pDelimiter OR t.N = 0) 
                )
--===== Do the actual split. The ISNULL/NULLIF combo handles the length for the final element when no delimiter is found.
 SELECT ItemNumber = ROW_NUMBER() OVER(ORDER BY s.N1),
        Item       = SUBSTRING(@pString,s.N1,ISNULL(NULLIF(CHARINDEX(@pDelimiter,@pString,s.N1),0)-s.N1,8000))
   FROM cteStart s
;
GO
--=====================================================================================================================
--      Conditionally drop and recreate a View that will allow us to use NEWID() within a function so we can make
--      Random numbers in a function and create a function that will create constrained randomized CSV element rows.
--=====================================================================================================================
--===== Conditionally drop the objects in the code below to make reruns easier
     IF OBJECT_ID('TempDB.dbo.iFunction'  ,'V' ) IS NOT NULL DROP VIEW     dbo.iFunction;
     IF OBJECT_ID('TempDB.dbo.CreateCsv8K','IF') IS NOT NULL DROP FUNCTION dbo.CreateCsv8K;
GO
 CREATE VIEW dbo.iFunction AS 
/**********************************************************************************************************************
 Purpose:
 This view is callable from UDF's which allows us to indirectly get a NEWID() within a function where we can't do such
 a thing directly in the function.  This view also solves the same problem for GETDATE().

 Usage:
 SELECT MyNewID FROM dbo.iFunction; --Returns a GUID
 SELECT MyDate  FROM dbo.iFunction; --Returns a Date


 Revision History:
 Rev 00 - 06 Jun 2004 - Jeff Moden - Initial creation
 Rev 01 - 06 Mar 2011 - Jeff Moden - Formalize code.  No logic changes. 
**********************************************************************************************************************/
SELECT MyNewID = NEWID(),
       MyDate  = GETDATE();
GO 
 CREATE FUNCTION dbo.CreateCsv8K
/**********************************************************************************************************************
 Purpose:
 Create a CSV table result with a programable number of rows, elements per row, minimum # of characters per element,
 and maximum characters per element.  The element size is random in nature constrained by the min and max characters
 per element.

 Usage:
 SELECT * FROM dbo.CreateCsv8K(@pNumberOfRows, @pNumberOfElementsPerRow, @pMinElementwidth, @pMaxElementWidth)

 Dependencies:
 1. View: dbo.iFunction (Produces a NEWID() usable from within a UDF)

 Programmer's Notes:
 1. The randomness of the elements prevents the delimiters for showing up in the same position for each row so that
    SQL Server won't figure that out and cache the information making some splitting techniques seem faster than they
    really are.
 2. No validation or constraints have been place on the input parameters so use with caution.  This code can generate
    a lot of data in a couple of heart beats.

 Revision History:
 Rev 00 - 11 May 2007 - Jeff Moden - Initial creation - Only returned one row and wasn't programmable.
 Rev 01 - 26 Jul 2009 - Jeff Moden - Added programmable variables but would only go to 20 characters wide.
 Rev 02 - 06 Mar 2011 - Jeff Moden - Converted to iTVF, added minimum element width, and made it so elements can be
                                     virtually any size.
**********************************************************************************************************************/
--===== Declare the I/0
        (
        @pNumberOfRows           INT,
        @pNumberOfElementsPerRow INT,
        @pMinElementwidth        INT,
        @pMaxElementWidth        INT
        )
RETURNS TABLE 
     AS
 RETURN
--===== This creates and populates a test table on the fly containing a
     -- sequential column and a randomly generated CSV Parameter column.
 SELECT TOP (@pNumberOfRows) --Controls the number of rows in the test table
        ISNULL(ROW_NUMBER() OVER (ORDER BY(SELECT NULL)),0) AS RowNum,
        CSV =
        (--==== This creates each CSV
         SELECT CAST(
                    STUFF( --=== STUFF get's rid of the leading comma
                         ( --=== This builds CSV row with a leading comma
                          SELECT TOP (@pNumberOfElementsPerRow) --Controls the number of CSV elements in each row
                                 ','
                               + LEFT(--==== Builds random length variable within element width constraints
                                      LEFT(REPLICATE('1234567890',CEILING(@pMaxElementWidth/10.0)), @pMaxElementWidth),
                                      ABS(CHECKSUM((SELECT MyNewID FROM dbo.iFunction)))
                                            % (@pMaxElementWidth - @pMinElementwidth + 1) + @pMinElementwidth
                                     )
                            FROM sys.All_Columns ac3            --Classic cross join pseudo-cursor
                           CROSS JOIN sys.All_Columns ac4       --can produce row sets up 16 million.
                           WHERE ac3.Object_ID <> ac1.Object_ID --Without this line, all rows would be the same.
                             FOR XML PATH('')
                         )
                    ,1,1,'')
                AS VARCHAR(8000))
        )
   FROM sys.All_Columns ac1      --Classic cross join pseudo-cursor
  CROSS JOIN sys.All_Columns ac2 --can produce row sets up 16 million rows
;
GO
--=====================================================================================================================
--      Conditionally drop and recreate the TestResults table
--=====================================================================================================================
--===== Conditionally drop and create the TestResults table
     IF OBJECT_ID('dbo.TestResults','U') IS NOT NULL DROP TABLE dbo.TestResults;

 CREATE TABLE dbo.TestResults
        (
        RowNum           INT IDENTITY(1,1) PRIMARY KEY CLUSTERED,
        SplitterName     VARCHAR(50),
        NumberOfRows     INT,
        NumberOfElements INT,
        MinElementLength    INT,
        MaxElementLength    INT,
        Duration         DECIMAL(9,5),
        MinLength        INT,
        AvgLength        INT,
        MaxLength        INT
        );
GO
--=====================================================================================================================
--      Conditionally drop and recreate the stored procedure that tests each function and records the results.
--=====================================================================================================================
--===== Conditionally drop and create the TestResults table
     IF OBJECT_ID('dbo.TestEachFunction','P') IS NOT NULL DROP PROCEDURE dbo.TestEachFunction;
GO
CREATE PROCEDURE dbo.TestEachFunction
/**********************************************************************************************************************
 Purpose:
 Given the number of rows andelements this testing is for, the stored procedure will test each of the split function
 for duration and record the results in an table called dbo.TestResults in the current DB (which should be TempDB).

 Revision History:
 Rev 00 - 10 Apr 2011 - Jeff Moden - Initial release for testing
**********************************************************************************************************************/
--===== Declare the I/O parameters
        @pNumberOfRows     INT,
        @pNumberOfElements INT,
        @pMinElementLength INT,
        @pMaxElementLength INT
     AS

--=====================================================================================================================
--      Presets
--=====================================================================================================================
--===== Suppress the auto-display of rowcounts for appearance and speed
    SET NOCOUNT ON;

--===== Declare some obviously named local variables
DECLARE @StartTime DATETIME,
        @EndTime   DATETIME,
        @Message   SYSNAME,
        @MinLength INT,
        @AvgLength INT,
        @MaxLength INT;

--===== Preset and display the current run message
 SELECT @Message = '========== ' 
                 + CAST(@pNumberOfRows     AS VARCHAR(10)) + ' Rows, '
                 + CAST(@pMinElementLength AS VARCHAR(10)) + ' MinElementSize, '
                 + CAST(@pMaxElementLength AS VARCHAR(10)) + ' MaxElementSize, '
                 + CAST(@pNumberOfElements AS VARCHAR(10)) + ' Elements '
                 + '==========';

RAISERROR(@Message,10,1) WITH NOWAIT;

--===== Calculate some statistics for the condition of the data
 SELECT @MinLength = MIN(DATALENGTH(CSV)), 
        @AvgLength = AVG(DATALENGTH(CSV)),
        @MaxLength = MAX(DATALENGTH(CSV))
   FROM dbo.Csv8K;

--===== Conditionally drop all test result tables =====================================================================
RAISERROR('Dropping temp tables...',10,1) WITH NOWAIT;
IF OBJECT_ID('tempdb..#SplitInchworm')       IS NOT NULL DROP TABLE #SplitInchworm;
IF OBJECT_ID('tempdb..#SplitNibbler')        IS NOT NULL DROP TABLE #SplitNibbler;
IF OBJECT_ID('tempdb..#SplitRCte')           IS NOT NULL DROP TABLE #SplitRCte;
IF OBJECT_ID('tempdb..#SplitXML')            IS NOT NULL DROP TABLE #SplitXML;
IF OBJECT_ID('tempdb..#OLDDelimitedSplit8K') IS NOT NULL DROP TABLE #OLDDelimitedSplit8K;
IF OBJECT_ID('tempdb..#DelimitedSplit8K')    IS NOT NULL DROP TABLE #DelimitedSplit8K;

--=====================================================================================================================
--      Run the tests
--=====================================================================================================================
--===== SplitInchWorm =================================================================================================
RAISERROR('Testing SplitInchWorm...',10,1) WITH NOWAIT;
--===== Start the timer
 SELECT @StartTime = GETDATE();

--===== Run the test
 SELECT csv.RowNum, split.ItemNumber, split.Item
   INTO #SplitInchWorm
   FROM dbo.CSV8K csv
  CROSS APPLY dbo.SplitInchWorm(csv.CSV,',') split;

--===== Stop the timer and record the test
 SELECT @EndTime = GETDATE();
 
 INSERT INTO dbo.TestResults
        (SplitterName, NumberOfRows, NumberOfElements, MinElementLength, MaxElementLength, Duration, MinLength, AvgLength, MaxLength)
 SELECT 'SplitInchWorm', 
        @pNumberOfRows, 
        @pNumberOfElements,
        @pMinElementLength,
        @pMaxElementLength,
        DATEDIFF(ms,@StartTime,@EndTime)/1000.0,
        MinLength = @MinLength,
        AvgLength = @AvgLength,
        MaxLength = @MaxLength
;

--===== SplitNibbler ==================================================================================================
RAISERROR('Testing SplitNibbler...',10,1) WITH NOWAIT;
--===== Start the timer
 SELECT @StartTime = GETDATE();

--===== Run the test
 SELECT csv.RowNum, split.ItemNumber, split.Item
   INTO #SplitNibbler
   FROM dbo.CSV8K csv
  CROSS APPLY dbo.SplitNibbler(csv.CSV,',') split;

--===== Stop the timer and record the test
 SELECT @EndTime = GETDATE();
 
 INSERT INTO dbo.TestResults
        (SplitterName, NumberOfRows, NumberOfElements, MinElementLength, MaxElementLength, Duration, MinLength, AvgLength, MaxLength)
 SELECT 'SplitNibbler', 
        @pNumberOfRows, 
        @pNumberOfElements, 
        @pMinElementLength,
        @pMaxElementLength,
        DATEDIFF(ms,@StartTime,@EndTime)/1000.0,
        MinLength = @MinLength,
        AvgLength = @AvgLength,
        MaxLength = @MaxLength
;

--===== SplitRCte =====================================================================================================
RAISERROR('Testing SplitRCte...',10,1) WITH NOWAIT;
--===== Start the timer
 SELECT @StartTime = GETDATE();

--===== Run the test
 SELECT csv.RowNum, split.ItemNumber, split.Item
   INTO #SplitRCte
   FROM dbo.CSV8K csv
  CROSS APPLY dbo.SplitRCte(csv.CSV,',') split
 OPTION (MAXRECURSION 32000);

--===== Stop the timer and record the test
 SELECT @EndTime = GETDATE();
 
 INSERT INTO dbo.TestResults
        (SplitterName, NumberOfRows, NumberOfElements, MinElementLength, MaxElementLength, Duration, MinLength, AvgLength, MaxLength)
 SELECT 'SplitRCte', 
        @pNumberOfRows, 
        @pNumberOfElements, 
        @pMinElementLength,
        @pMaxElementLength,
        DATEDIFF(ms,@StartTime,@EndTime)/1000.0,
        MinLength = @MinLength,
        AvgLength = @AvgLength,
        MaxLength = @MaxLength
;

--===== SplitXML ======================================================================================================
RAISERROR('Testing SplitXML...',10,1) WITH NOWAIT;
--===== Start the timer
 SELECT @StartTime = GETDATE();

--===== Run the test
 SELECT csv.RowNum, split.ItemNumber, split.Item
   INTO #SplitXML
   FROM dbo.CSV8K csv
  CROSS APPLY dbo.SplitXML(csv.CSV,',') split;

--===== Stop the timer and record the test
 SELECT @EndTime = GETDATE();
 
 INSERT INTO dbo.TestResults
        (SplitterName, NumberOfRows, NumberOfElements, MinElementLength, MaxElementLength, Duration, MinLength, AvgLength, MaxLength)
 SELECT 'SplitXML', 
        @pNumberOfRows, 
        @pNumberOfElements, 
        @pMinElementLength,
        @pMaxElementLength,
        DATEDIFF(ms,@StartTime,@EndTime)/1000.0,
        MinLength = @MinLength,
        AvgLength = @AvgLength,
        MaxLength = @MaxLength
;

--===== OLDDelimitedSplit8K ===========================================================================================
RAISERROR('Testing OLDDelimitedSplit8K...',10,1) WITH NOWAIT;
--===== Start the timer
 SELECT @StartTime = GETDATE();

--===== Run the test
 SELECT csv.RowNum, split.ItemNumber, split.Item
   INTO #OLDDelimitedSplit8K
   FROM dbo.CSV8K csv
  CROSS APPLY dbo.OLDDelimitedSplit8K(csv.CSV,',') split;

--===== Stop the timer and record the test
 SELECT @EndTime = GETDATE();
 
 INSERT INTO dbo.TestResults
        (SplitterName, NumberOfRows, NumberOfElements, MinElementLength, MaxElementLength, Duration, MinLength, AvgLength, MaxLength)
 SELECT 'OLDDelimitedSplit8K', 
        @pNumberOfRows, 
        @pNumberOfElements, 
        @pMinElementLength,
        @pMaxElementLength,
        DATEDIFF(ms,@StartTime,@EndTime)/1000.0,
        MinLength = @MinLength,
        AvgLength = @AvgLength,
        MaxLength = @MaxLength
;

--===== DelimitedSplit8K (NEW) ========================================================================================
RAISERROR('Testing DelimitedSplit8K...',10,1) WITH NOWAIT;
--===== Start the timer
 SELECT @StartTime = GETDATE();

--===== Run the test
 SELECT csv.RowNum, split.ItemNumber, split.Item
   INTO #DelimitedSplit8K
   FROM dbo.CSV8K csv
  CROSS APPLY dbo.DelimitedSplit8K(csv.CSV,',') split;

--===== Stop the timer and record the test
 SELECT @EndTime = GETDATE();
 
 INSERT INTO dbo.TestResults
        (SplitterName, NumberOfRows, NumberOfElements, MinElementLength, MaxElementLength, Duration, MinLength, AvgLength, MaxLength)
 SELECT 'DelimitedSplit8K', 
        @pNumberOfRows, 
        @pNumberOfElements, 
        @pMinElementLength,
        @pMaxElementLength,
        DATEDIFF(ms,@StartTime,@EndTime)/1000.0,
        MinLength = @MinLength,
        AvgLength = @AvgLength,
        MaxLength = @MaxLength
;

RAISERROR('==============================================================================================',10,1) WITH NOWAIT;
GO
--=====================================================================================================================
--      We're ready to rock.  Now, run all the tests automatically
--=====================================================================================================================
--===== Alert the operator as to how to check the run status
 SELECT Info = 'Please make sure you''re in the grid mode when running these tests.' UNION ALL 
 SELECT 'Click on the "Messages" tab if you want to check that the code is still running.';

--===== Declare some obviously named variables
DECLARE @SQL VARCHAR(MAX);

--===== Suppress the auto-display of rowcounts
    SET NOCOUNT ON;

--===== Create a "control" CTE and build all of the test commands from that
   WITH cteControl (NumberOfRows, NumberOfElements, MinElementLength, MaxElementLength)
     AS
        (
        --===== 1 to 10 characters per element
         SELECT 1000,    1, 1, 10 UNION ALL
         SELECT 1000,    2, 1, 10 UNION ALL
         SELECT 1000,    4, 1, 10 UNION ALL
         SELECT 1000,    8, 1, 10 UNION ALL
         SELECT 1000,   16, 1, 10 UNION ALL
         SELECT 1000,   32, 1, 10 UNION ALL
         SELECT 1000,   64, 1, 10 UNION ALL
         SELECT 1000,  128, 1, 10 UNION ALL
         SELECT 1000,  256, 1, 10 UNION ALL
         SELECT 1000,  512, 1, 10 UNION ALL
         SELECT 1000, 1150, 1, 10 UNION ALL
        --===== 10 to 20 characters per element
         SELECT 1000,    1, 10, 20 UNION ALL
         SELECT 1000,    2, 10, 20 UNION ALL
         SELECT 1000,    4, 10, 20 UNION ALL
         SELECT 1000,    8, 10, 20 UNION ALL
         SELECT 1000,   16, 10, 20 UNION ALL
         SELECT 1000,   32, 10, 20 UNION ALL
         SELECT 1000,   64, 10, 20 UNION ALL
         SELECT 1000,  128, 10, 20 UNION ALL
         SELECT 1000,  256, 10, 20 UNION ALL
         SELECT 1000,  480, 10, 20 UNION ALL
        --===== 20 to 30 characters per element
         SELECT 1000,    1, 20, 30 UNION ALL
         SELECT 1000,    2, 20, 30 UNION ALL
         SELECT 1000,    4, 20, 30 UNION ALL
         SELECT 1000,    8, 20, 30 UNION ALL
         SELECT 1000,   16, 20, 30 UNION ALL
         SELECT 1000,   32, 20, 30 UNION ALL
         SELECT 1000,   64, 20, 30 UNION ALL
         SELECT 1000,  128, 20, 30 UNION ALL
         SELECT 1000,  256, 20, 30 UNION ALL
         SELECT 1000,  290, 20, 30 UNION ALL
        --===== 30 to 40 characters per element
         SELECT 1000,    1, 30, 40 UNION ALL
         SELECT 1000,    2, 30, 40 UNION ALL
         SELECT 1000,    4, 30, 40 UNION ALL
         SELECT 1000,    8, 30, 40 UNION ALL
         SELECT 1000,   16, 30, 40 UNION ALL
         SELECT 1000,   32, 30, 40 UNION ALL
         SELECT 1000,   64, 30, 40 UNION ALL
         SELECT 1000,  128, 30, 40 UNION ALL
         SELECT 1000,  210, 30, 40 UNION ALL
        --===== 40 to 50 characters per element
         SELECT 1000,    1, 40, 50 UNION ALL
         SELECT 1000,    2, 40, 50 UNION ALL
         SELECT 1000,    4, 40, 50 UNION ALL
         SELECT 1000,    8, 40, 50 UNION ALL
         SELECT 1000,   16, 40, 50 UNION ALL
         SELECT 1000,   32, 40, 50 UNION ALL
         SELECT 1000,   64, 40, 50 UNION ALL
         SELECT 1000,  128, 40, 50 UNION ALL
         SELECT 1000,  165, 40, 50
        )
--===== Dynamically build all of the test commands from the above
 SELECT @SQL = ISNULL(@SQL,'')+
'
IF OBJECT_ID(''dbo.Csv8K'',''U'') IS NOT NULL DROP TABLE dbo.Csv8K;
 SELECT * 
   INTO dbo.Csv8K 
   FROM dbo.CreateCsv8K
        ('+CAST(NumberOfRows     AS VARCHAR(10))+', '
          +CAST(NumberOfElements AS VARCHAR(10))+', '
          +CAST(MinElementLength AS VARCHAR(10))+', '
          +CAST(MaxElementLength AS VARCHAR(10))+'); --# of Rows, # of Elements, MIN element length, MAX element length
   EXEC dbo.TestEachFunction '+CAST(NumberOfRows     AS VARCHAR(10)) +', '
                              +CAST(NumberOfElements AS VARCHAR(10))+', '
                              +CAST(MinElementLength    AS VARCHAR(10))+', '
                              +CAST(MaxElementLength    AS VARCHAR(10))+';
'
   FROM cteControl
--PRINT @SQL
--===== Run the tests
   EXEC (@SQL);

--===== Display the results in the grid so the tester can copy and paste to a spreadsheet.
 SELECT * FROM dbo.TestResults;

--===== Tell the operator what to do next
 SELECT Info = 'Test Complete.  Please copy and paste the results from the grid above to a spreadsheet' UNION ALL
 SELECT 'and email the spreadsheet to jbmoden@ameritech.net.' UNION ALL
 SELECT 'Thank you for helping me on these tests.';

 PRINT 
'Test Complete.  Please copy and paste the results from the 2nd Grid to a spreadsheet 
and email the spreadsheet to jbmoden@ameritech.net.
Thank you for helping me on these tests.'
GO
--=====================================================================================================================
--      Housekeeping (Left table TestResults standing just in case)
--=====================================================================================================================
   DROP TABLE
        dbo.Csv8K;

   DROP VIEW
        dbo.iFunction;

   DROP PROCEDURE
        dbo.TestEachFunction;

   DROP FUNCTION 
        dbo.CreateCsv8K,
        dbo.DelimitedSplit8K,
        dbo.OLDDelimitedSplit8K,
        dbo.SplitInchWorm,
        dbo.SplitNibbler,
        dbo.SplitRCte,
        dbo.SplitXML;
GO
